module simpleDB	//	Small database program to manipulate a simple database
import basicDB
import deltaPrint, deltaIOState

Start :: *World -> *World
Start world
	=	snd (StartIO [MenuSystem [file, comm]] {old=initState,current=initState} initIO world)
where
	file = PullDownMenu DontCareId "File" Able
	 		[ MenuItem DontCareId "Read new..."		 (Key 'o') Able ReadNew
	 		, MenuItem DontCareId "Save As..."		 (Key 's') Able SaveRecords
			, MenuItem DontCareId "Print"			 (Key 'p') Able Print
	 		, MenuSeparator
	 		, MenuItem DontCareId "Quit"			 (Key 'q') Able Quit
	 		]
	comm = PullDownMenu DontCareId "Commands" Able
	 		[ MenuItem DontCareId "Show Records"	 (Key 'r') Able ShowRecords
	 		, MenuItem DontCareId "Edit & Query..."	 (Key 'e') Able ShowEditDialog
	 		, MenuItem DontCareId "Change Set Up..." (Key 'u') Able ShowFieldDialog
	 		, MenuSeparator
	 		, MenuItem DontCareId "Undo"			 (Key 'z') Able Undo
	 		]
		
	initIO		= [	ReadDataBase, ShowRecords, ShowEditDialog		]
	closeIO		= [ CloseWindows [RecordWindowId], closeDbDialogs	]
	
	ReadNew s io = seqIO initIO (s, seq closeIO io)
	Undo {old,current} io = seqIO [ShowRecords, ShowEditDialog] ({old=current,current=old},seq closeIO io)
	
	initState	= { records		= []
				  , descriptor	= []
				  , selection	= 0
				  , query		= []
				  , name		= ""
				  , fw			= 0
				  , dw			= 0
				  }

// fopen for use with accFiles
fopen2 fileName mode files
	:==	((ok, file), files2)
	where
		(ok, file, files2)
			=	fopen fileName mode files

//	The CallBack and initialisation Functions of the menu:

ReadDataBase :: *DataBase IO -> (*DataBase, IO)
ReadDataBase db io
 #	(done,dbname,db,io)		= SelectInputFile db io
 |	not done				= (db,io)
 #	((open,dbfile),io)		= accFiles (fopen2 dbname FReadText) io
 |	not open				= (db,Beep io)
 #	(descr,dbfile)			= FReadDescr dbfile
	(recs, dbfile)			= FReadRecords (inc (length descr)) dbfile
	(close,io)				= accFiles (fclose dbfile) io
 |	not close				= (db,Beep io)
 #	{old,current}			= db
 |	otherwise				= (	{ old=current
 								, current=
 									{ current
 									& records	 = recs
 									, descriptor = descr
 									, query		 = repeatn (length descr) (AS "")
 									, selection	 = 0
 									, name		 = dbname
 									, fw		 = MaxWidth DbFont.font (map toString (flatten recs))
 									, dw		 = MaxWidth DbFont.font (map toString descr)
 									}
 							 	}
							  , io
							  )
where
	FReadDescr file
	 #	(nroffields,file)	= FReadStrippedLine file
		(fields,file)		= seqList (repeatn (toInt nroffields) FReadStrippedLine) file
	 =	([AttDesc field STRING \\ field <- fields],file)
	
	FReadRecords nroflines file
	 |	sfend file			= ([], file)
	 #	([_:record],file)	= seqList (repeatn nroflines FReadStrippedLine) file
		(records,	file)	= FReadRecords nroflines file
	 =	([map AS record: records], file)
	
	FReadStrippedLine file
	 #	(line, file)		= freadline file
	 =	(line%(0,size line - 2),file)		// strip "\n"

ShowRecords :: *DataBase IO -> (*DataBase, IO)
ShowRecords db=:{current=state=:{records,name}} io
 =	(db,OpenWindows [window] io)
where
	domain = DbPictureDomain state 0 (max (length records) 1)
	((left,top),(right,bottom)) = domain
	stripDirs name = toString (last (splitby DirSeparator (fromString name)))
	window = ScrollWindow
				RecordWindowId
				(5,5)
				(stripDirs name)
				(ScrollBar (Thumb left) (Scroll DbFont.width))
				(ScrollBar (Thumb top) (Scroll DbFont.height))
				domain
				MinDbDomainSize
				(right - left,bottom - top)
				UpdateRecordWindow
				[Mouse Able MouseSelectItem]

ShowFieldDialog :: *DataBase IO -> (*DataBase, IO)
ShowFieldDialog db=:{current={descriptor=d}} io 
 |	isEmpty d	= inputdialog "Give first field" InputBoxWidth (\input->FieldChangeIO (add (-1) input "")) db io
 |	otherwise	= (db,OpenDialog fielddialog (CloseDialog EdDialogId io))
with
	fielddialog	= CommandDialog FieldDialogId "Change Set Up" [] addId 
					[ StaticText DontCareId Left "Select Field..."
		 			, RadioButtons selectId Left (Columns 1) firstRadioId
		 				(radioitems firstRadioId (map toString d))
					, DialogButton deleteId Left "Delete" Able (DeleteField getselectedfield)
					, DialogButton moveId (RightTo deleteId) "Move" Able (MoveField getselectedfield)
					, DialogButton renameId Left "Rename" Able (RenameField getselectedfield)
					, DialogButton addId  (Below moveId) "Add New" Able (AddField getselectedfield)
					]
	getselectedfield dialoginfo = GetSelectedRadioItemId selectId dialoginfo - firstRadioId
	[deleteId,moveId,renameId,addId,selectId,firstRadioId:_] = [0..]

SaveRecords :: *DataBase IO -> (*DataBase, IO)
SaveRecords db=:{current=state=:{name,descriptor,records}} io
 #	(ok,dbname,db,io)  	= SelectOutputFile "Save As: " name db io
 |	not ok				= (db,io)
 #	((open,dbfile),io)	= accFiles (fopen2 dbname FWriteText) io
 |	not open			= (db, Beep io)
 #	(close,io)		 	= accFiles (fclose (seq (writedescriptor++writerecords) dbfile)) io
 |	not close			= (db, Beep io)
 |	otherwise			= ({db & old=state,current={state & name = dbname}}, io)
where
	writedescriptor	= [ fwritei (length descriptor)
					  , FWriteList descriptor
					  ]
	writerecords	= map FWriteList records
	FWriteList l	= fwrites (foldl (\s t -> s+++toString t+++"\n") "\n" l)


Print :: *DataBase IO -> (*DataBase, IO)
Print db=:{current={records,descriptor,name,dw}} io
	#	(defaulPS, io)	= defaultPrintSetup io
		(_,{ s=db, io=io }) = print True True pages defaulPS { s=db,io=io }
	= (db,io)
  where
	pages {printSetup, jobInfo={range=(first,last), copies}} picture
		= (map print_page groups_printed, picture)
	  where
		{page=(width,height)}	= getPageDimensions printSetup True
		(fAscent, fDescent, _, fLeading) = FontMetrics DbFont.font
		line_height = DbFont.height
		record_height = (1+(length descriptor))*line_height
		table_height = height - 2*line_height
		nr_records_per_page = table_height/record_height		// fa: first approximation
		groups = group_by nr_records_per_page records
		groups_printed = flatten (repeatn copies (groups % (first-1,last-1)))
	
		print_page record_group picture
			# line_pos_y = 3*line_height/2
			  picture = seq [	SetFont DbFont.font,
			  					MovePenTo (0,fAscent+fLeading),
								DrawString name,
								MovePenTo (0,line_pos_y),
								LinePenTo (width,line_pos_y)
							] picture
			= seq [	print_record (n*record_height+2*line_height) descriptor field_values 
					\\ n<-[0..] & field_values<-record_group ]
				  picture
	
		print_record y field_names field_values picture
			= seq [ draw_row s1 s2 n \\ s1<-field_names & s2<-field_values & n<-[1..]] picture
		  where
			draw_row (AttDesc s1 STRING) (AS s2) n picture
				# baseline = y+n*line_height-fDescent
				= seq [	MovePenTo (0,baseline),
						DrawString s1,
						MovePenTo (dw,baseline),
						DrawString (":"+++s2)
					  ] picture
	
		group_by :: !Int [x] -> [[x]];
		group_by n [] = [];
		group_by n l = [(take n l ) : (group_by n (drop n l))]; 
		  
Quit :: (*DataBase -> *(IO -> (*DataBase, IO)))
Quit = warnOK ["Unsaved changes will be lost","Quit Now?"] (\s io -> (s, QuitIO io))

// Field set up changes

FieldChangeIO :: (State -> State) *DataBase IO -> (*DataBase,IO)
FieldChangeIO changefun {current} io = UpdateDbDomain {old=current,current=changefun current} (closeDbDialogs io)
	
AddField :: (DialogInfo -> Int) DialogInfo *DataBase IO -> (*DataBase, IO)
AddField getfield dialoginfo db=:{current={descriptor}} io 
 =	inputdialog infotext InputBoxWidth (ReadDefault fId) db io
where
	infotext = "Add after '"+++toString (descriptor!!fId)+++"' new field"
	fId		 = getfield dialoginfo

ReadDefault :: Int String *DataBase IO -> (*DataBase, IO)
ReadDefault fId fName db io
 =	inputdialog infotext InputBoxWidth (\val->FieldChangeIO (add fId fName val)) db io
where
	infotext = "Default value for '"+++toString fName+++"'"

RenameField :: (DialogInfo -> Int) DialogInfo *DataBase IO -> (*DataBase, IO)
RenameField getfield dialoginfo db=:{current} io
 =	inputdialog infotext InputBoxWidth (\input->FieldChangeIO (rename fieldtorename input)) db io
where
	infotext	   = "Rename '"+++toString (current.descriptor!!fieldtorename)+++"' to"
	fieldtorename  = getfield dialoginfo

MoveField :: (DialogInfo -> Int) DialogInfo *DataBase IO -> (*DataBase, IO)
MoveField getfield dialoginfo db=:{current={descriptor=d}} io
 =	(db,OpenDialog movedialog io)
where
	fieldtomove = getfield dialoginfo
	movedialog  
	 = CommandDialog moveDialogId "Move Field" [] okId 
		[ StaticText   infoId	Left ("Move '"+++(toString (d!!fieldtomove))+++ "' before: ")
		, RadioButtons selectId	Left (Rows (inc (length d))) firstRadioId
			(radioitems firstRadioId (map toString d++[""]))
		, DialogButton cancelId	Left Cancel Able cancel
		, DialogButton okId		(RightTo cancelId) "Move" Able (ok (move fieldtomove))
		]
	[moveDialogId,cancelId,okId,infoId, selectId,firstRadioId:_] = [0..]	 
	ok mvf dlginfo s io 
	 =	FieldChangeIO (mvf destinationfield) s (CloseDialog moveDialogId io)
	where
		destinationfield = GetSelectedRadioItemId selectId dlginfo - firstRadioId
		
DeleteField :: (DialogInfo -> Int) DialogInfo *DataBase IO -> (*DataBase, IO)
DeleteField getfield dialoginfo db io
 =	warnCancel ["This will also remove the attribute values.","Are you sure?"] (FieldChangeIO (delete (getfield dialoginfo))) db io

add :: Int String String State -> State
add afterfield fieldname value state=:{records,descriptor,query,dw}
 =	{ state
	& records	 = map (ins (AS value)) records
	, descriptor = ins (AttDesc fieldname STRING) descriptor
	, query		 = ins (AS "") query
	, dw		 = max (MaxWidth DbFont.font [fieldname]) dw
	}
where
	ins x ys	 = insertAt (inc afterfield) x ys

rename :: Int String State -> State
rename selectedfield newfieldname s=:{descriptor=d} 
 =	{ s
	& descriptor = newdescr
	, dw		 = MaxWidth DbFont.font (map toString newdescr)
	}
where
	newdescr = updateAt selectedfield (AttDesc newfieldname STRING) d

move :: Int Int State -> State
move sf df s=:{records=rs,descriptor=d,query=q}
 =	{ s
	& records	 = map (moveinlist sf df) rs
	, descriptor = moveinlist sf df d
	, query		 = moveinlist sf df q
	}

delete :: Int State -> State
delete i s=:{records=rs,descriptor=d,query=q} 
 =	{ s
	& records	 = newrs
	, descriptor = newdescr
	, query		 = removeAt i q
	, dw		 = MaxWidth DbFont.font (map toString newdescr)
	, fw		 = nfw
	}
where
	newrs    = map (removeAt i) rs
	newdescr = removeAt i d
	nfw      = MaxWidth DbFont.font (map toString (flatten newrs))

//	Handling mouse clicks in database window

MouseSelectItem	:: MouseState *DataBase IO -> (*DataBase, IO)
MouseSelectItem ((_,xPos), ButtonDown, _) db=:{current=state} io
 |	isEmpty state.records = (db, io)
 = ( {db & old=state,current={state & selection=newSelection}}
   , ChangeSelection state state.selection newSelection io
   )
where
	newSelection = toRecCo state.descriptor xPos
MouseSelectItem _ db io = (db, io)

// Various useful functions

closeDbDialogs io = seq (map CloseDialog [FieldDialogId,EdDialogId]) io 

radioitems firstid titles = [RadioItem id t Able (\ _ x -> x) \\ id <- [firstid..] & t <- titles]
		
// functions that should be library functions

seqIO fs = seq (map uncurry fs)		// should be in deltaEventIO, will be obsolete with new IO-library

Cancel		:==	"Cancel"
OK			:==	"OK"

inputdialog :: String Measure (String -> .(*a -> .((*IOState *a) -> *(*a,*IOState *a)))) .b *(IOState *a) -> *(.b,*IOState *a);
inputdialog name width fun s io
 =	(s,OpenDialog dialogdef io)
where
	dialogdef	= CommandDialog dlgId name [] okId
					[ StaticText nameId Left (name+++": ")
					, EditText inputId (RightTo nameId) width 1 ""
					, DialogButton cancelId (Below inputId) Cancel Able cancel
					, DialogButton okId (RightTo cancelId) OK Able ok
					]
	ok dlginfo s io	= fun (GetEditText inputId dlginfo) s (CloseDialog dlgId io)
	[dlgId,nameId,inputId,cancelId,okId:_]	= [1000..]

/* from I.5 */
warnCancel :: [a] .(MenuFunction *s (IOState *s)) *s (IOState *s) 
												-> (*s,IOState *s) | toString a
warnCancel info fun s io
 #	(choiceId,s,io)    = OpenNotice warningdef s io
 |	choiceId==cancelId = (s,io)
 |	otherwise 	       = fun s io
where
	warningdef = Notice (map toString info) (NoticeButton cancelId "Cancel")
											[NoticeButton okId "OK"]
	[cancelId,okId:_] = [0..]

/*warning on function to be applied: default OK, from I.5 */	
warnOK :: [a] .(MenuFunction *s (IOState *s)) *s (IOState *s) 
												-> (*s,IOState *s) | toString a
warnOK info fun s io
 #	(choiceId,s,io)    = OpenNotice warningdef s io
 |	choiceId==cancelId = (s,io)
 |	otherwise 	       = fun s io
where
	warningdef = Notice (map toString info) (NoticeButton okId "OK")
											[NoticeButton cancelId "Cancel"]
	[cancelId,okId:_] = [0..]

cancel _ s io = (s, CloseActiveDialog io)
